<?php
/**
 * FUZIION Plugin Updater Class
 *
 * Bundled with FUZIION Redirects. Handles checking for updates and updating
 */
class Fuziion_Plugin_Updater {

    private $api_url;
    private $plugin_slug;
    private $plugin_file;
    private $current_version;

    /**
     * Constructor.
     *
     * @param string $api_url
     * @param string $plugin_slug
     * @param string $plugin_file
     * @param string $current_version
     */
    public function __construct( $api_url, $plugin_slug, $plugin_file, $current_version ) {
        $this->api_url = trailingslashit( $api_url );
        $this->plugin_slug = $plugin_slug;
        $this->plugin_file = $plugin_file;
        $this->current_version = $current_version;

        add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_for_update' ) );
        add_filter( 'plugins_api', array( $this, 'plugins_api_call' ), 10, 3 );
    }

    /**
     * Check for updates against the custom API.
     *
     * @param object $transient The update transient.
     * @return object Modified transient.
     */
    public function check_for_update( $transient ) {
        if ( empty( $transient->checked ) ) {
            return $transient;
        }

        $response = $this->call_api( 'plugins/' . $this->plugin_slug . '/version');

        if ( $response && isset( $response->new_version ) && version_compare( $this->current_version, $response->new_version, '<' ) ) {
            $plugin_data = new stdClass();
            $plugin_data->slug = $this->plugin_slug;
            $plugin_data->new_version = $response->new_version;
            $plugin_data->url = $response->url ?? '';
            $plugin_data->package = $this->api_url . 'plugin/' . $this->plugin_slug . '/download';
            $plugin_data->tested = $response->tested ?? null;
            $plugin_data->requires = $response->requires ?? null;
            $plugin_data->requires_php = $response->requires_php ?? null;
            $plugin_data->sections = $response->sections ?? array();

            $transient->response[ $this->plugin_file ] = $plugin_data;
        }

        return $transient;
    }

    /**
     * Intercept the plugins_api call to provide plugin details from our API.
     *
     * @param false|object|array $result 
     * @param string $action
     * @param object $args
     * @return object $result
     */
    public function plugins_api_call( $result, $action, $args ) {
        if ( 'plugin_information' === $action && isset( $args->slug ) && $args->slug === $this->plugin_slug ) {
            $response = $this->call_api( 'plugin/' . $this->plugin_slug . '/info');
            if ( $response ) {
                $result = $response;
                if ( ! isset( $result->slug ) ) {
                    $result->slug = $this->plugin_slug;
                }
                if ( ! isset( $result->name ) ) {
                    $result->name = $this->plugin_slug;
                }
                if ( ! isset( $result->version ) ) {
                    $result->version = $this->current_version;
                }
                if ( ! isset( $result->last_updated ) ) {
                    $result->last_updated = date( 'Y-m-d H:i:s' );
                }
                if ( ! isset( $result->download_link ) ) {
                    $result->download_link = $this->api_url . 'plugin/' . $this->plugin_slug . '/download';
                }
            }
        }
        return $result;
    }

    /**
     * Make a request to the custom API.
     *
     * @param string $endpoint
     * @return object|false
     */
    private function call_api( $endpoint ) {
        $url = $this->api_url . $endpoint;
        $args = array(
            'timeout'     => 15,
            'redirection' => 5,
            'blocking'    => true,
            'headers'     => array(
                'Accept' => 'application/json',
            ),
            'sslverify'   => false,
        );

        $request = wp_remote_get( $url, $args );

        if ( is_wp_error( $request ) || wp_remote_retrieve_response_code( $request ) !== 200 ) {
            return false;
        }

        return json_decode( wp_remote_retrieve_body( $request ) );
    }
}
