<?php
/**
 * Handles the "add redirect" form submission on the admin page.
 *
 * @package FuziionRedirects
 */

namespace FuziionRedirects\Admin;

use FuziionRedirects\RedirectRepository;
use FuziionRedirects\PathNormalizer;

defined('ABSPATH') || exit;

class AdminFormHandler
{
    private const VALID_TYPES = [301, 302, 307, 308];

    public function __construct(
        private RedirectRepository $repository,
        private PathNormalizer $normalizer
    ) {
    }

    /**
     * Run on admin_init: verify nonce and capability, then add redirect if posted.
     *
     * @return void
     */
    public function handleSubmission(): void
    {
        if (!isset($_POST['fuziion_redirects_nonce']) || !wp_verify_nonce(
            sanitize_text_field(wp_unslash($_POST['fuziion_redirects_nonce'])),
            'fuziion_redirects_action'
        )) {
            return;
        }

        if (!current_user_can('manage_options')) {
            return;
        }

        if (isset($_POST['add_redirect'])) {
            $this->addRedirect();
        }
    }

    /**
     * Validate input, check duplicate, then add one redirect and save.
     *
     * @return void
     */
    private function addRedirect(): void
    {
        $from = isset($_POST['redirect_from']) ? sanitize_text_field(wp_unslash($_POST['redirect_from'])) : '';
        $to = isset($_POST['redirect_to']) ? sanitize_text_field(wp_unslash($_POST['redirect_to'])) : '';
        $type = isset($_POST['redirect_type']) ? (int) $_POST['redirect_type'] : 301;

        if ($from === '' || $to === '') {
            add_settings_error(
                'fuziion_redirects',
                'empty_fields',
                __('Both fields are required.', 'fuziion-redirects'),
                'error'
            );
            return;
        }

        $from = $this->normalizer->normalize($from);
        $to = filter_var($to, FILTER_VALIDATE_URL) ? $to : $this->normalizer->normalize($to);

        if (!in_array($type, self::VALID_TYPES, true)) {
            $type = 301;
        }

        $redirects = $this->repository->getRedirects();

        foreach ($redirects as $redirect) {
            if (($redirect['from'] ?? '') === $from) {
                add_settings_error(
                    'fuziion_redirects',
                    'duplicate',
                    __('A redirect for this path already exists.', 'fuziion-redirects'),
                    'error'
                );
                return;
            }
        }

        $redirects[] = [
            'from' => $from,
            'to' => $to,
            'type' => $type,
            'created' => current_time('mysql'),
        ];

        $this->repository->saveRedirects($redirects);

        add_settings_error(
            'fuziion_redirects',
            'success',
            __('Redirect added successfully.', 'fuziion-redirects'),
            'success'
        );
    }
}
