<?php
/**
 * Renders the redirects admin page.
 *
 * @package FuziionRedirects
 */

namespace FuziionRedirects\Admin;

use FuziionRedirects\RedirectRepository;

defined('ABSPATH') || exit;

class AdminPageRenderer
{
    public function __construct(
        private RedirectRepository $repository
    ) {
    }

    /**
     * Output the admin page HTML.
     *
     * @return void
     */
    public function render(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have permission to view this page.', 'fuziion-redirects'));
        }

        $redirects = $this->repository->getRedirects();
        settings_errors('fuziion_redirects');
        ?>
        <div class="wrap fuziion-redirects-wrap">
            <header class="fuziion-redirects-header">
                <h1 class="fuziion-redirects-title"><?php echo esc_html(get_admin_page_title()); ?></h1>
                <p class="fuziion-redirects-desc"><?php _e('Set up redirects from old URLs to new pages or URLs. Add a redirect below.', 'fuziion-redirects'); ?></p>
            </header>

            <section class="fuziion-redirects-card fuziion-redirects-form-card">
                <h2 class="fuziion-redirects-card-title"><?php _e('Add redirect', 'fuziion-redirects'); ?></h2>
                <form method="post" action="" class="fuziion-redirects-form">
                    <?php wp_nonce_field('fuziion_redirects_action', 'fuziion_redirects_nonce'); ?>
                    <div class="fuziion-redirects-form-row">
                        <div class="fuziion-redirects-field">
                            <label for="redirect_from" class="fuziion-redirects-label"><?php _e('From', 'fuziion-redirects'); ?></label>
                            <input
                                type="text"
                                id="redirect_from"
                                name="redirect_from"
                                class="fuziion-redirects-input"
                                placeholder="/old-page"
                                required
                                autocomplete="off"
                            />
                        </div>
                        <span class="fuziion-redirects-arrow" aria-hidden="true">→</span>
                        <div class="fuziion-redirects-field">
                            <label for="redirect_to" class="fuziion-redirects-label"><?php _e('To', 'fuziion-redirects'); ?></label>
                            <input
                                type="text"
                                id="redirect_to"
                                name="redirect_to"
                                class="fuziion-redirects-input"
                                placeholder="/new-page or https://..."
                                required
                                autocomplete="off"
                            />
                        </div>
                        <div class="fuziion-redirects-field fuziion-redirects-field-type">
                            <label for="redirect_type" class="fuziion-redirects-label"><?php _e('Type', 'fuziion-redirects'); ?></label>
                            <select id="redirect_type" name="redirect_type" class="fuziion-redirects-select" title="<?php esc_attr_e('301 = permanent, 302 = temporary', 'fuziion-redirects'); ?>">
                                <option value="301" title="<?php esc_attr_e('Permanent', 'fuziion-redirects'); ?>">301</option>
                                <option value="302" title="<?php esc_attr_e('Temporary', 'fuziion-redirects'); ?>">302</option>
                                <option value="307" title="<?php esc_attr_e('Temporary (preserve method)', 'fuziion-redirects'); ?>">307</option>
                                <option value="308" title="<?php esc_attr_e('Permanent (preserve method)', 'fuziion-redirects'); ?>">308</option>
                            </select>
                        </div>
                        <div class="fuziion-redirects-field fuziion-redirects-field-submit">
                            <label class="fuziion-redirects-label fuziion-redirects-label-invisible"><?php _e('Add', 'fuziion-redirects'); ?></label>
                            <button type="submit" name="add_redirect" class="fuziion-redirects-btn fuziion-redirects-btn-primary">
                                <?php esc_attr_e('Add', 'fuziion-redirects'); ?>
                            </button>
                        </div>
                    </div>
                    <p class="fuziion-redirects-hint"><?php _e('From: the path visitors use now. To: the new path or full URL. 301 = permanent, 302 = temporary.', 'fuziion-redirects'); ?></p>
                </form>
            </section>

            <section class="fuziion-redirects-card fuziion-redirects-list-card">
                <div class="fuziion-redirects-list-head">
                    <h2 class="fuziion-redirects-card-title"><?php _e('Existing redirects', 'fuziion-redirects'); ?></h2>
                    <?php if (!empty($redirects)): ?>
                        <span class="fuziion-redirects-count"><?php echo esc_html((string) count($redirects)); ?> <?php _e('redirects', 'fuziion-redirects'); ?></span>
                    <?php endif; ?>
                </div>
                <?php if (empty($redirects)): ?>
                    <div class="fuziion-redirects-empty" data-empty="1">
                        <div class="fuziion-redirects-empty-icon" aria-hidden="true"></div>
                        <p class="fuziion-redirects-empty-title"><?php _e('No redirects yet', 'fuziion-redirects'); ?></p>
                        <p class="fuziion-redirects-empty-text"><?php _e('Add your first redirect above. Visitors will be redirected automatically.', 'fuziion-redirects'); ?></p>
                    </div>
                <?php else: ?>
                    <div class="fuziion-redirects-table-wrap">
                        <table class="fuziion-redirects-table">
                            <thead>
                                <tr>
                                    <th class="fuziion-redirects-th-from"><?php _e('From', 'fuziion-redirects'); ?></th>
                                    <th class="fuziion-redirects-th-to"><?php _e('To', 'fuziion-redirects'); ?></th>
                                    <th class="fuziion-redirects-th-type"><?php _e('Type', 'fuziion-redirects'); ?></th>
                                    <th class="fuziion-redirects-th-actions"><?php _e('Action', 'fuziion-redirects'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($redirects as $index => $redirect): ?>
                                    <tr class="fuziion-redirects-row" data-index="<?php echo esc_attr((string) $index); ?>">
                                        <td class="fuziion-redirects-cell-from">
                                            <code class="fuziion-redirects-path"><?php echo esc_html($redirect['from'] ?? ''); ?></code>
                                        </td>
                                        <td class="fuziion-redirects-cell-to">
                                            <code class="fuziion-redirects-path"><?php echo esc_html($redirect['to'] ?? ''); ?></code>
                                        </td>
                                        <td class="fuziion-redirects-cell-type">
                                            <span class="fuziion-redirects-badge fuziion-redirects-badge-<?php echo esc_attr((string) ($redirect['type'] ?? 301)); ?>">
                                                <?php echo esc_html((string) ($redirect['type'] ?? 301)); ?>
                                            </span>
                                        </td>
                                        <td class="fuziion-redirects-cell-actions">
                                            <button type="button" class="fuziion-redirects-btn fuziion-redirects-btn-ghost fuziion-redirects-btn-delete delete-redirect" data-index="<?php echo esc_attr((string) $index); ?>">
                                                <?php _e('Delete', 'fuziion-redirects'); ?>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </section>
        </div>
        <?php
    }
}
