<?php
/**
 * Dependency injection container with constructor autoloading.
 *
 * Binds services and resolves class dependencies via reflection.
 * Unbound class names are auto-resolved from constructor type hints.
 *
 * @package FuziionRedirects
 */

namespace FuziionRedirects;

use ReflectionClass;
use ReflectionParameter;

defined('ABSPATH') || exit;

class Container
{
    /**
     * Singleton instances (service id => instance).
     *
     * @var array<string, object>
     */
    private array $instances = [];

    /**
     * Factory closures (service id => callable).
     *
     * @var array<string, callable>
     */
    private array $bindings = [];

    /**
     * Bind a service to the container.
     *
     * @param string   $id      Service identifier (e.g. class name or string id).
     * @param callable $factory Factory closure: function(Container $c): object.
     * @return void
     */
    public function bind(string $id, callable $factory): void
    {
        $this->bindings[$id] = $factory;
    }

    /**
     * Bind a singleton. First resolve is cached.
     *
     * @param string   $id      Service identifier.
     * @param callable $factory Factory closure: function(Container $c): object.
     * @return void
     */
    public function singleton(string $id, callable $factory): void
    {
        $this->bind($id, function (Container $c) use ($id, $factory) {
            if (!isset($this->instances[$id])) {
                $this->instances[$id] = $factory($c);
            }
            return $this->instances[$id];
        });
    }

    /**
     * Get a service from the container.
     * Resolves unbound class names via constructor injection.
     *
     * @param string $id Service identifier or class name.
     * @return object
     */
    public function get(string $id): object
    {
        if (isset($this->instances[$id])) {
            return $this->instances[$id];
        }

        if (isset($this->bindings[$id])) {
            $instance = $this->bindings[$id]($this);
            $this->instances[$id] = $instance;
            return $instance;
        }

        return $this->resolve($id);
    }

    /**
     * Resolve a class by instantiating it and injecting constructor dependencies.
     *
     * @param string $class Fully qualified class name.
     * @return object
     */
    private function resolve(string $class): object
    {
        if (!class_exists($class)) {
            throw new \InvalidArgumentException("Class does not exist: {$class}");
        }

        $reflection = new ReflectionClass($class);
        $constructor = $reflection->getConstructor();

        if ($constructor === null) {
            $instance = $reflection->newInstance();
        } else {
            $args = [];
            foreach ($constructor->getParameters() as $param) {
                $args[] = $this->resolveParameter($param);
            }
            $instance = $reflection->newInstanceArgs($args);
        }

        $this->instances[$class] = $instance;
        return $instance;
    }

    /**
     * Resolve a constructor parameter from the container.
     *
     * @param ReflectionParameter $param
     * @return mixed
     */
    private function resolveParameter(ReflectionParameter $param): mixed
    {
        $type = $param->getType();

        if ($type === null || $type->isBuiltin()) {
            if ($param->isDefaultValueAvailable()) {
                return $param->getDefaultValue();
            }
            throw new \InvalidArgumentException(
                "Cannot resolve parameter \${$param->getName()} of type " . ($type ? $type->getName() : 'mixed')
            );
        }

        $typeName = $type->getName();
        return $this->get($typeName);
    }

    /**
     * Check if a service is bound or already resolved.
     *
     * @param string $id
     * @return bool
     */
    public function has(string $id): bool
    {
        return isset($this->instances[$id]) || isset($this->bindings[$id]) || class_exists($id);
    }
}
