<?php
/**
 * Handles front-end redirect matching and execution.
 *
 * @package FuziionRedirects
 */

namespace FuziionRedirects;

defined('ABSPATH') || exit;

class RedirectHandler
{
    public function __construct(
        private RedirectRepository $repository,
        private PathNormalizer $normalizer
    ) {
    }

    /**
     * Run on template_redirect: match request and redirect if found.
     *
     * @return void
     */
    public function handleRedirects(): void
    {
        if (is_admin()) {
            return;
        }

        $request_uri = $_SERVER['REQUEST_URI'] ?? '';
        $request_path = parse_url($request_uri, PHP_URL_PATH);

        if ($request_path === null || $request_path === false) {
            return;
        }

        $request_path = $this->normalizer->normalize($request_path);
        $redirects = $this->repository->getRedirects();

        foreach ($redirects as $redirect) {
            if (($redirect['from'] ?? '') !== $request_path) {
                continue;
            }

            $to = $redirect['to'] ?? '';

            if (!filter_var($to, FILTER_VALIDATE_URL)) {
                $to = home_url($to);
            }

            $query_string = parse_url($request_uri, PHP_URL_QUERY);
            if ($query_string !== null && $query_string !== false) {
                $to .= (strpos($to, '?') !== false ? '&' : '?') . $query_string;
            }

            $status_code = isset($redirect['type']) ? (int) $redirect['type'] : 301;
            wp_redirect($to, $status_code);
            exit;
        }
    }
}
